/* 
 * DHCP library functions.
 * Copyright (C) 2003, 2004 Mondru AB.
 *
 * The contents of this file may be used under the terms of the GNU
 * General Public License Version 2, provided that the above copyright
 * notice and this permission notice is included in all copies or
 * substantial portions of the software.
 * 
 */


/* Usage
 *
 * The library is initialised by calling dhcp_new(), which
 * initialises a dhcp_t struct that is used for all subsequent calls
 * to the library. Ressources are freed by calling dhcp_free().
 * 
 */

/* TODO
 *
 * Port to FreeBSD. 
 * - Mainly concerns Ethernet stuff.
 * 
 * Move EAPOL stuff to separate files
 *
 * Change static memory allocation to malloc
 * - Mainly concerns newconn() and freeconn()
 * - Wait until code is bug free.
 */

#include <stdlib.h>
#include <syslog.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <stdint.h> /* ISO C99 types */
#include <arpa/inet.h>
#include <linux/if.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#include <net/if_arp.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include "../config.h"
#include "syserr.h"
#include "iphash.h"
#include "dhcp.h"

#ifdef NAIVE
const static int paranoid = 0; /* Trust that the program has no bugs */
#else
const static int paranoid = 1; /* Check for errors which cannot happen */
#endif


/**
 * dhcp_ip_check()
 * Generates an IPv4 header checksum.
 **/
int dhcp_ip_check(struct dhcp_ippacket_t *pack) {
  int i;
  uint32_t sum = 0;
  pack->iph.check = 0;
  for (i=0; i<(pack->iph.ihl * 2); i++) {
    sum += ((uint16_t*) &pack->iph)[i];
  }
  while (sum>>16)
    sum = (sum & 0xFFFF)+(sum >> 16);
  pack->iph.check = ~sum;
  return 0;
}

/**
 * dhcp_udp_check()
 * Generates an UDP header checksum.
 **/
int dhcp_udp_check(struct dhcp_fullpacket_t *pack) {
  int i;
  uint32_t sum = 0;
  int udp_len = ntohs(pack->udph.len);

  pack->udph.check = 0;
  
  if (udp_len > DHCP_UDP_HLEN + DHCP_LEN)
    return -1; /* Packet too long */

  /* Sum UDP header and payload */
  for (i=0; i<(udp_len/2); i++) {
    sum += ((uint16_t*) &pack->udph)[i];
  }

  /* Sum any uneven payload octet */
  if (udp_len & 0x01) {
    sum += ((uint8_t*) &pack->udph)[udp_len-1];
  }

  /* Sum both source and destination address */
  for (i=0; i<4; i++) {
    sum += ((uint16_t*) &pack->iph.saddr)[i];
  }

  /* Sum both protocol and udp_len (again) */
  sum = sum + pack->udph.len + ((pack->iph.protocol<<8)&0xFF00);

  while (sum>>16)
    sum = (sum & 0xFFFF)+(sum >> 16);

  pack->udph.check = ~sum;

  return 0;
}


/**
 * dhcp_tcp_check()
 * Generates an TCP header checksum.
 **/
int dhcp_tcp_check(struct dhcp_ippacket_t *pack, int length) {
  int i;
  uint32_t sum = 0;
  struct dhcp_tcphdr_t *tcph;
  int tcp_len;

  if (ntohs(pack->iph.tot_len) > (length - DHCP_ETH_HLEN))
    return -1; /* Wrong length of packet */

  tcp_len = ntohs(pack->iph.tot_len) - pack->iph.ihl * 4;

  if (tcp_len < 20) /* TODO */
    return -1; /* Packet too short */

  tcph = (struct dhcp_tcphdr_t*) pack->payload;
  tcph->check = 0;

  /* Sum TCP header and payload */
  for (i=0; i<(tcp_len/2); i++) {
    sum += ((uint16_t*) pack->payload)[i];
  }

  /* Sum any uneven payload octet */
  if (tcp_len & 0x01) {
    sum += ((uint8_t*) pack->payload)[tcp_len-1];
  }

  /* Sum both source and destination address */
  for (i=0; i<4; i++) {
    sum += ((uint16_t*) &pack->iph.saddr)[i];
  }

  /* Sum both protocol and tcp_len */
  sum = sum + htons(tcp_len) + ((pack->iph.protocol<<8)&0xFF00);

  while (sum>>16)
    sum = (sum & 0xFFFF)+(sum >> 16);

  tcph->check = ~sum;

  return 0;
}


int dhcp_sifflags(char const *devname, int flags) {
  struct ifreq ifr;
  int fd;
  
  memset (&ifr, '\0', sizeof (ifr));
  ifr.ifr_flags = flags;
  strncpy(ifr.ifr_name, devname, IFNAMSIZ);
  ifr.ifr_name[IFNAMSIZ-1] = 0; /* Make sure to terminate */
  if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "socket() failed");
  }
  if (ioctl(fd, SIOCSIFFLAGS, &ifr)) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "ioctl(SIOCSIFFLAGS) failed");
    close(fd);
    return -1;
  }
  close(fd);
  return 0;
}

int dhcp_gifflags(char const *devname, int *flags) {
  struct ifreq ifr;
  int fd;
  
  memset (&ifr, '\0', sizeof (ifr));
  strncpy(ifr.ifr_name, devname, IFNAMSIZ);
  ifr.ifr_name[IFNAMSIZ-1] = 0; /* Make sure to terminate */
  if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "socket() failed");
  }
  if (ioctl(fd, SIOCGIFFLAGS, &ifr)) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "ioctl(SIOCSIFFLAGS) failed");
    close(fd);
    return -1;
  }
  close(fd);
  *flags = ifr.ifr_flags;

  return 0;
}

int dhcp_setaddr(char const *devname,
		 struct in_addr *addr,
		 struct in_addr *dstaddr,
		 struct in_addr *netmask)
{
  struct ifreq   ifr;
  int flags;
  int fd;

  memset (&ifr, '\0', sizeof (ifr));
  ifr.ifr_addr.sa_family = AF_INET;
  ifr.ifr_dstaddr.sa_family = AF_INET;
  ifr.ifr_netmask.sa_family = AF_INET;
  strncpy(ifr.ifr_name, devname, IFNAMSIZ);
  ifr.ifr_name[IFNAMSIZ-1] = 0; /* Make sure to terminate */

  /* Create a channel to the NET kernel. */
  if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "socket() failed");
    return -1;
  }

  if (addr) { /* Set the interface address */
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = addr->s_addr;
    if (ioctl(fd, SIOCSIFADDR, (void *) &ifr) < 0) {
      if (errno != EEXIST) {
	sys_err(LOG_ERR, __FILE__, __LINE__, errno,
		"ioctl(SIOCSIFADDR) failed");
      }
      else {
	sys_err(LOG_WARNING, __FILE__, __LINE__, errno,
		"ioctl(SIOCSIFADDR): Address already exists");
      }
      close(fd);
      return -1;
    }
  }

  if (dstaddr) { /* Set the destination address */
    ((struct sockaddr_in *) &ifr.ifr_dstaddr)->sin_addr.s_addr = 
      dstaddr->s_addr;
    if (ioctl(fd, SIOCSIFDSTADDR, (caddr_t) &ifr) < 0) {
      sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	      "ioctl(SIOCSIFDSTADDR) failed");
      close(fd);
      return -1; 
    }
  }

  if (netmask) { /* Set the netmask */
    ((struct sockaddr_in *) &ifr.ifr_netmask)->sin_addr.s_addr = 
      netmask->s_addr;
    if (ioctl(fd, SIOCSIFNETMASK, (void *) &ifr) < 0) {
      sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	      "ioctl(SIOCSIFNETMASK) failed");
      close(fd);
      return -1;
    }
  }

  close(fd);
  dhcp_gifflags(devname, &flags);
  flags |= IFF_UP | IFF_RUNNING;
  dhcp_sifflags(devname, flags);
  return 0;
}



/**
 * dhcp_open_eth()
 * Opens an Ethernet interface. As an option the interface can be set in
 * promisc mode. If not null macaddr and ifindex are filled with the
 * interface mac address and index
 **/
int dhcp_open_eth(char const *ifname, uint16_t protocol, int promisc,
		  int usemac, unsigned char *macaddr, int *ifindex) {
  int fd;
  int option=1;
  struct ifreq ifr;
  struct packet_mreq mr;
  struct sockaddr_ll sa;
  
  memset(&ifr, 0, sizeof(ifr));

  /* Create socket */
  if ((fd = socket(PF_PACKET, SOCK_RAW, htons(protocol))) < 0) {
    if (errno == EPERM) {
      sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	      "Cannot create raw socket. Must be root.");
    }
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "socket(domain=%d, protocol=%lx, protocol=%d) failed",
	    PF_PACKET, SOCK_RAW, protocol);
  }


  /* Enable reception and transmission of broadcast frames */
  if (setsockopt(fd, SOL_SOCKET, SO_BROADCAST, &option, sizeof(option)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "setsockopt(s=%d, level=%d, optname=%d, optlen=%d) failed",
	    fd, SOL_SOCKET, SO_BROADCAST, sizeof(option));
  }
  

  /* Get the MAC address of our interface */
  if ((!usemac) && (macaddr)) {
    strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
    if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) {
      sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	      "ioctl(d=%d, request=%d) failed",
	      fd, SIOCGIFHWADDR);
    }
    memcpy(macaddr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);
    if (ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "Not Ethernet: %.16s", ifname);
    }
    
    if (macaddr[0] & 0x01) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0, 
	      "Ethernet has broadcast or multicast address: %.16s", ifname);
    }
  }


  /* Verify that MTU = ETH_DATA_LEN */
  strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
  if (ioctl(fd, SIOCGIFMTU, &ifr) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "ioctl(d=%d, request=%d) failed",
	    fd, SIOCGIFMTU);
  }
  if (ifr.ifr_mtu != ETH_DATA_LEN) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "MTU does not match EHT_DATA_LEN: %d %d", 
	    ifr.ifr_mtu, ETH_DATA_LEN);
  }

  
  /* Get ifindex */
  strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
  if (ioctl(fd, SIOCGIFINDEX, &ifr) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "ioctl(SIOCFIGINDEX) failed");
  }
  if (ifindex)
    *ifindex = ifr.ifr_ifindex;
  
  
  /* Set interface in promisc mode */
  if (promisc) {
    memset(&mr,0,sizeof(mr));
    mr.mr_ifindex = ifr.ifr_ifindex;
    mr.mr_type =  PACKET_MR_PROMISC;
    if(setsockopt(fd, SOL_PACKET, PACKET_ADD_MEMBERSHIP,
		  (char *)&mr, sizeof(mr)) < 0) {
      sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	      "setsockopt(s=%d, level=%d, optname=%d, optlen=%d) failed",
	      fd, SOL_SOCKET, PACKET_ADD_MEMBERSHIP, sizeof(mr));
    }
  }


  /* Bind to particular interface */
  memset(&sa, 0, sizeof(sa));
  sa.sll_family = AF_PACKET;
  sa.sll_protocol = htons(protocol);
  sa.sll_ifindex = ifr.ifr_ifindex;
  if (bind(fd, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "bind(sockfd=%d) failed", fd);
  }
  return fd;
}



/**
 * lookup()
 * Generates a 32 bit hash.
 * Based on public domain code by Bob Jenkins
 * It should be one of the best hash functions around in terms of both
 * statistical properties and speed. It is NOT recommended for cryptographic
 * purposes.
 **/
unsigned long int static lookup( k, length, level)
register unsigned char *k;         /* the key */
register unsigned long int length; /* the length of the key */
register unsigned long int level; /* the previous hash, or an arbitrary value*/
{

#define mix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

  typedef  unsigned long  int  ub4;   /* unsigned 4-byte quantities */
  typedef  unsigned       char ub1;   /* unsigned 1-byte quantities */
  register unsigned long int a,b,c,len;
  
  /* Set up the internal state */
  len = length;
  a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
  c = level;           /* the previous hash value */
  
  /*---------------------------------------- handle most of the key */
  while (len >= 12)
    {
      a += (k[0] +((ub4)k[1]<<8) +((ub4)k[2]<<16) +((ub4)k[3]<<24));
      b += (k[4] +((ub4)k[5]<<8) +((ub4)k[6]<<16) +((ub4)k[7]<<24));
      c += (k[8] +((ub4)k[9]<<8) +((ub4)k[10]<<16)+((ub4)k[11]<<24));
      mix(a,b,c);
      k += 12; len -= 12;
    }
  
  /*------------------------------------- handle the last 11 bytes */
  c += length;
  switch(len)              /* all the case statements fall through */
    {
    case 11: c+=((ub4)k[10]<<24);
    case 10: c+=((ub4)k[9]<<16);
    case 9 : c+=((ub4)k[8]<<8);
      /* the first byte of c is reserved for the length */
    case 8 : b+=((ub4)k[7]<<24);
    case 7 : b+=((ub4)k[6]<<16);
    case 6 : b+=((ub4)k[5]<<8);
    case 5 : b+=k[4];
    case 4 : a+=((ub4)k[3]<<24);
    case 3 : a+=((ub4)k[2]<<16);
    case 2 : a+=((ub4)k[1]<<8);
    case 1 : a+=k[0];
      /* case 0: nothing left to add */
    }
  mix(a,b,c);
  /*-------------------------------------------- report the result */
  return c;
}


/**
 * dhcp_hash()
 * Generates a 32 bit hash based on a mac address
 **/
unsigned long int dhcp_hash(uint8_t *hwaddr) {
  return lookup(hwaddr, DHCP_ETH_ALEN, 0);
}


/**
 * dhcp_hashinit()
 * Initialises hash tables
 **/
int dhcp_hashinit(struct dhcp_t *this, int listsize) {
  /* Determine hashlog */
  for ((this)->hashlog = 0; 
       ((1 << (this)->hashlog) < listsize);
       (this)->hashlog++);
  
  /* Determine hashsize */
  (this)->hashsize = 1 << (this)->hashlog;
  (this)->hashmask = (this)->hashsize -1;
  
  /* Allocate hash table */
  if (!((this)->hash = calloc(sizeof(struct dhcp_conn_t), (this)->hashsize))){
    /* Failed to allocate memory for hash members */
    return -1;
  }
  return 0;
}


/**
 * dhcp_hashadd()
 * Adds a connection the hash table
 **/
int dhcp_hashadd(struct dhcp_t *this, struct dhcp_conn_t *conn) {
  uint32_t hash;
  struct dhcp_conn_t *p;
  struct dhcp_conn_t *p_prev = NULL; 

  /* Insert into hash table */
  hash = dhcp_hash(conn->hismac) & this->hashmask;
  for (p = this->hash[hash]; p; p = p->nexthash)
    p_prev = p;
  if (!p_prev)
    this->hash[hash] = conn;
  else 
    p_prev->nexthash = conn;
  return 0; /* Always OK to insert */
}


/**
 * dhcp_hashdel()
 * Removes a connection the hash table
 **/
int dhcp_hashdel(struct dhcp_t *this, struct dhcp_conn_t *conn) {
  uint32_t hash;
  struct dhcp_conn_t *p;
  struct dhcp_conn_t *p_prev = NULL; 

  /* Find in hash table */
  hash = dhcp_hash(conn->hismac) & this->hashmask;
  for (p = this->hash[hash]; p; p = p->nexthash) {
    if (p == conn) {
      break;
    }
    p_prev = p;
  }

  if ((paranoid) && (p!= conn)) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "Tried to delete connection not in hash table");
  }

  if (!p_prev)
    this->hash[hash] = p->nexthash;
  else
    p_prev->nexthash = p->nexthash;
  
  return 0;
}


/**
 * dhcp_hashget()
 * Uses the hash tables to find a connection based on the mac address.
 * Returns -1 if not found.
 **/
int dhcp_hashget(struct dhcp_t *this, struct dhcp_conn_t **conn,
		 uint8_t *hwaddr) {
  struct dhcp_conn_t *p;
  uint32_t hash;

  /* Find in hash table */
  hash = dhcp_hash(hwaddr) & this->hashmask;
  for (p = this->hash[hash]; p; p = p->nexthash) {
    if ((!memcmp(p->hismac, hwaddr, DHCP_ETH_ALEN)) && (p->inuse)) {
      *conn = p;
      return 0;
    }
  }
  *conn = NULL;
  return -1; /* Address could not be found */
}


/**
 * dhcp_validate()
 * Valides reference structures of connections. 
 * Returns the number of active connections
 **/
int dhcp_validate(struct dhcp_t *this)
{
  int used = 0;
  int unused = 0;
  struct dhcp_conn_t *conn;
  struct dhcp_conn_t *hash_conn;
  
  /* Count the number of used connections */
  conn = this->firstusedconn;
  while (conn) {
    if (!conn->inuse) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "Connection with inuse == 0!");
    }
    dhcp_hashget(this, &hash_conn, conn->hismac);
    if (conn != hash_conn) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "Connection could not be found by hashget!");
    }
    used ++;
    conn = conn->next;
  }
  
  /* Count the number of unused connections */
  conn = this->firstfreeconn;
  while (conn) {
    if (conn->inuse) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "Connection with inuse != 0!");
    }
    unused ++;
    conn = conn->next;
  }

  if (this->numconn != (used + unused)) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "The number of free and unused connections does not match!");
    if (this->debug) {
      printf("used %d unused %d\n", used, unused);
      conn = this->firstusedconn;
      while (conn) {
	printf("%.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n", 
	       conn->hismac[0], conn->hismac[1], conn->hismac[2],
	       conn->hismac[3], conn->hismac[4], conn->hismac[5]);
	conn = conn->next;
      }
    }
  }
  
  return used;
}


/**
 * dhcp_initconn()
 * Initialises connection references
 **/
int dhcp_initconn(struct dhcp_t *this)
{
  int n;
  this->firstusedconn = NULL; /* Redundant */
  this->lastusedconn  = NULL; /* Redundant */

  for (n=0; n<this->numconn; n++) {
    this->conn[n].inuse = 0; /* Redundant */
    if (n == 0) {
      this->conn[n].prev = NULL; /* Redundant */
      this->firstfreeconn = &this->conn[n];

    }
    else {
      this->conn[n].prev = &this->conn[n-1];
      this->conn[n-1].next = &this->conn[n];
    }
    if (n == (this->numconn-1)) {
      this->conn[n].next = NULL; /* Redundant */
      this->lastfreeconn  = &this->conn[n];
    }
  }

  if (paranoid) dhcp_validate(this);

  return 0;
}

/**
 * dhcp_newconn()
 * Allocates a new connection from the pool. 
 * Returns -1 if unsuccessful.
 **/
int dhcp_newconn(struct dhcp_t *this, struct dhcp_conn_t **conn, 
		 uint8_t *hwaddr)
{

  if (this->debug) 
    printf("DHCP newconn: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n", 
	   hwaddr[0], hwaddr[1], hwaddr[2],
	   hwaddr[3], hwaddr[4], hwaddr[5]);


  if (!this->firstfreeconn) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "Out of free connections");
    return -1;
  }

  *conn = this->firstfreeconn;

  /* Remove from link of free */
  if (this->firstfreeconn->next) {
    this->firstfreeconn->next->prev = NULL;
    this->firstfreeconn = this->firstfreeconn->next;
  }
  else { /* Took the last one */
    this->firstfreeconn = NULL; 
    this->lastfreeconn = NULL;
  }

  /* Initialise structures */
  memset(*conn, 0, sizeof(**conn));

  /* Insert into link of used */
  if (this->firstusedconn) {
    this->firstusedconn->prev = *conn;
    (*conn)->next = this->firstusedconn;
  }
  else { /* First insert */
    this->lastusedconn = *conn;
  }

  this->firstusedconn = *conn;

  (*conn)->inuse = 1;
  (*conn)->parent = this;

  /* Application specific initialisations */
  memcpy((*conn)->hismac, hwaddr, DHCP_ETH_ALEN);
  memcpy((*conn)->ourmac, this->hwaddr, DHCP_ETH_ALEN);
  gettimeofday(&(*conn)->lasttime, NULL);
  dhcp_hashadd(this, *conn);
  
  if (paranoid) dhcp_validate(this);

  /* Inform application that connection was created */
  if (this ->cb_connect)
    this ->cb_connect(*conn);
  
  return 0; /* Success */
}


/**
 * dhcp_freeconn()
 * Returns a connection to the pool. 
 **/
int dhcp_freeconn(struct dhcp_conn_t *conn)
{
  /* TODO: Always returns success? */

  struct dhcp_t *this = conn->parent;

  /* Tell application that we disconnected */
  if (this->cb_disconnect)
    this->cb_disconnect(conn);

  if (this->debug)
    printf("DHCP freeconn: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n", 
	   conn->hismac[0], conn->hismac[1], conn->hismac[2],
	   conn->hismac[3], conn->hismac[4], conn->hismac[5]);



  /* Application specific code */
  /* First remove from hash table */
  dhcp_hashdel(this, conn);


  /* Remove from link of used */
  if ((conn->next) && (conn->prev)) {
    conn->next->prev = conn->prev;
    conn->prev->next = conn->next;
  }
  else if (conn->next) { /* && prev == 0 */
    conn->next->prev = NULL;
    this->firstusedconn = conn->next;
  }
  else if (conn->prev) { /* && next == 0 */
    conn->prev->next = NULL;
    this->lastusedconn = conn->prev;
  }
  else { /* if ((next == 0) && (prev == 0)) */
    this->firstusedconn = NULL;
    this->lastusedconn = NULL;
  }

  /* Initialise structures */
  memset(conn, 0, sizeof(*conn));

  /* Insert into link of free */
  if (this->firstfreeconn) {
    this->firstfreeconn->prev = conn;
  }
  else { /* First insert */
    this->lastfreeconn = conn;
  }

  conn->next = this->firstfreeconn;
  this->firstfreeconn = conn;

  if (paranoid) dhcp_validate(this);

  return 0;
}


/**
 * dhcp_checkconn()
 * Checks connections to see if the lease has expired
 **/
int dhcp_checkconn(struct dhcp_t *this)
{
  struct dhcp_conn_t *conn;
  struct timeval now;

  gettimeofday(&now, NULL);
  now.tv_sec -= this->lease;

  conn = this->firstusedconn;
  while (conn) {
    if (timercmp(&now, &conn->lasttime, >)) {
      if (this->debug) printf("DHCP timeout: Removing connection\n");
      dhcp_freeconn(conn);
      return 0; /* Returning after first deletion */
    }
    conn = conn->next;
  }
  
  return 0;
}


/* API Functions */

/**
 * dhcp_version()
 * Returns the current version of the program
 **/
const char* dhcp_version()
{
  return VERSION;
}


/**
 * dhcp_new()
 * Allocates a new instance of the library
 **/

int
dhcp_new(struct dhcp_t **dhcp, int debug,
	 int numconn, char *interface,
	 int usemac, uint8_t *mac, 
	 int promisc, struct in_addr *listen, int lease, int allowdyn,
	 struct in_addr *uamlisten, uint16_t uamport,
	 int useeapol, struct in_addr *authip, int authiplen, int anydns,
	 struct in_addr *uamokip, int uamokiplen, struct in_addr *uamokaddr,
	 struct in_addr *uamokmask, int uamoknetlen) {

  
  /* sys_err(LOG_INFO, __FILE__, __LINE__, 0,
     "Dhcp started");*/

  int i;
  struct in_addr noaddr;

  if (!(*dhcp = calloc(sizeof(struct dhcp_t), 1))) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "calloc() failed");
    return -1;
  }

  (*dhcp)->numconn = numconn;

  if (!((*dhcp)->conn = calloc(sizeof(struct dhcp_conn_t), numconn))) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "calloc() failed");
    free(*dhcp);
    return -1;
  }

  (*dhcp)->authiplen = authiplen;

  if (!((*dhcp)->authip = calloc(sizeof(struct in_addr), authiplen))) {
    sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	    "calloc() failed");
    free((*dhcp)->conn);
    free(*dhcp);
    return -1;
  }
  memcpy((*dhcp)->authip, authip, sizeof(struct in_addr) * authiplen);

  (*dhcp)->debug = debug;

  dhcp_initconn(*dhcp);

  strncpy((*dhcp)->devname, interface, IFNAMSIZ);
  (*dhcp)->devname[IFNAMSIZ] = 0;

  /* Bring network interface UP and RUNNING if currently down */
  dhcp_gifflags((*dhcp)->devname, &(*dhcp)->devflags);
  if (!((*dhcp)->devflags & IFF_UP) || !((*dhcp)->devflags & IFF_RUNNING)) {
    dhcp_sifflags((*dhcp)->devname, (*dhcp)->devflags | IFF_NOARP);
    memset(&noaddr, 0, sizeof(noaddr));
    dhcp_setaddr((*dhcp)->devname, &noaddr, NULL, NULL);
  }
  
  if (usemac) memcpy(((*dhcp)->hwaddr), mac, DHCP_ETH_ALEN);
  if (((*dhcp)->fd = 
       dhcp_open_eth(interface, DHCP_ETH_IP, promisc, usemac,
		     ((*dhcp)->hwaddr),
		     &((*dhcp)->ifindex))) < 0) 
    {
      free((*dhcp)->conn);
      free(*dhcp);
      return -1; /* Error reporting done in dhcp_open_eth */
    }

  if (usemac) memcpy(((*dhcp)->arp_hwaddr), mac, DHCP_ETH_ALEN);
  if (((*dhcp)->arp_fd = 
       dhcp_open_eth(interface, DHCP_ETH_ARP, promisc, usemac,
		     ((*dhcp)->arp_hwaddr),
		     &((*dhcp)->arp_ifindex))) < 0) 
    {
      close((*dhcp)->fd);
      free((*dhcp)->conn);
      free(*dhcp);
      return -1; /* Error reporting done in dhcp_open_eth */
    }

  if (!useeapol) {
    (*dhcp)->eapol_fd = 0;
  }
  else {
    if (usemac) memcpy(((*dhcp)->eapol_hwaddr), mac, DHCP_ETH_ALEN);
    if (((*dhcp)->eapol_fd = 
	 dhcp_open_eth(interface, DHCP_ETH_EAPOL, promisc, usemac,
		       ((*dhcp)->eapol_hwaddr), &((*dhcp)->eapol_ifindex))) < 0) {
      close((*dhcp)->fd);
      close((*dhcp)->arp_fd);
      free((*dhcp)->conn);
      free(*dhcp);
      return -1; /* Error reporting done in eapol_open_eth */
    }
  }
  

  if (dhcp_hashinit(*dhcp, (*dhcp)->numconn))
    return -1; /* Failed to allocate hash tables */


  /* Initialise various variables */
  (*dhcp)->ourip.s_addr = listen->s_addr;
  (*dhcp)->lease = lease;
  (*dhcp)->promisc = promisc;
  (*dhcp)->usemac = usemac;
  (*dhcp)->allowdyn = allowdyn;
  (*dhcp)->uamlisten.s_addr = uamlisten->s_addr;
  (*dhcp)->uamport = uamport;
  (*dhcp)->anydns = anydns;

  
  /* Initialise call back functions */
  (*dhcp)->cb_data_ind = 0;
  (*dhcp)->cb_eap_ind = 0;
  (*dhcp)->cb_request = 0;
  (*dhcp)->cb_disconnect = 0;
  (*dhcp)->cb_connect = 0;

  /* Make hash table for allowed domains */
  if ((!uamokip) || (uamokiplen==0)) {
    (*dhcp)->iphashm = NULL;
    (*dhcp)->iphash = NULL;
  }
  else {
    if (!((*dhcp)->iphashm = calloc(uamokiplen, sizeof(struct iphashm_t)))) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "calloc() failed");
      return -1;
    }

    for (i=0; i< uamokiplen; i++) {
      (*dhcp)->iphashm[i].addr = uamokip[i];
    }
    
    iphash_new(&(*dhcp)->iphash, (*dhcp)->iphashm, uamokiplen);
  }

  /* Copy allowed networks */
  if ((!uamokaddr) || (!uamokmask) || (uamoknetlen==0)) {
    (*dhcp)->uamokaddr = NULL;
    (*dhcp)->uamokmask = NULL;
    (*dhcp)->uamoknetlen = 0;
  }
  else {
    (*dhcp)->uamoknetlen = uamoknetlen;
    if (!((*dhcp)->uamokaddr = calloc(uamoknetlen, sizeof(struct in_addr)))) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "calloc() failed");
      return -1;
    }
    if (!((*dhcp)->uamokmask = calloc(uamoknetlen, sizeof(struct in_addr)))) {
      sys_err(LOG_ERR, __FILE__, __LINE__, 0,
	      "calloc() failed");
      return -1;
    }
    memcpy((*dhcp)->uamokaddr, uamokaddr, uamoknetlen);
    memcpy((*dhcp)->uamokmask, uamokmask, uamoknetlen);
  }
  
  return 0;
}

/**
 * dhcp_free()
 * Releases ressources allocated to the instance of the library
 **/
int dhcp_free(struct dhcp_t *dhcp) {

  if (dhcp->iphash) iphash_free(dhcp->iphash);
  if (dhcp->iphashm) free(dhcp->iphashm);
  if (dhcp->authip) free(dhcp->authip);
  if (dhcp->uamokaddr) free(dhcp->uamokaddr);
  if (dhcp->uamokmask) free(dhcp->uamokmask);
  dhcp_sifflags(dhcp->devname, dhcp->devflags);
  close(dhcp->fd);
  close(dhcp->arp_fd);
  close(dhcp->eapol_fd);
  free(dhcp->conn);
  free(dhcp);
  return 0;
}

/**
 * dhcp_timeout()
 * Need to call this function at regular intervals to clean up old connections.
 **/
int
dhcp_timeout(struct dhcp_t *this)
{
  if (paranoid) 
    dhcp_validate(this);

  dhcp_checkconn(this);
  
  return 0;
}

/**
 * dhcp_timeleft()
 * Use this function to find out when to call dhcp_timeout()
 * If service is needed after the value given by tvp then tvp
 * is left unchanged.
 **/
struct timeval*
dhcp_timeleft(struct dhcp_t *this, struct timeval *tvp)
{
  return tvp;
}


/**
 * dhcp_doDNAT()
 * Change destination address to authentication server.
 **/
int dhcp_doDNAT(struct dhcp_conn_t *conn, 
		struct dhcp_ippacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_tcphdr_t *tcph = (struct dhcp_tcphdr_t*) pack->payload;
  struct dhcp_udphdr_t *udph = (struct dhcp_udphdr_t*) pack->payload;
  int i;

  /* Was it a DNS request? */
  if (((this->anydns) ||
       (pack->iph.daddr == conn->dns1.s_addr) ||
       (pack->iph.daddr == conn->dns2.s_addr)) &&
      (pack->iph.protocol == DHCP_IP_UDP) &&
      (udph->dst == htons(DHCP_DNS)))
    return 0; 

  /* Was it a http or https request for authentication server? */
  for (i = 0; i<this->authiplen; i++) {
    if ((pack->iph.daddr == this->authip[i].s_addr) &&
	(pack->iph.protocol == DHCP_IP_TCP) &&
	((tcph->dst == htons(DHCP_HTTP)) ||
	 (tcph->dst == htons(DHCP_HTTPS))))
      return 0; /* Destination was authentication server */
  }

  /* Was it a request for local redirection server? */
  if ((pack->iph.daddr == this->uamlisten.s_addr) &&
      (pack->iph.protocol == DHCP_IP_TCP) &&
      (tcph->dst == htons(this->uamport)))
    return 0; /* Destination was local redir server */

  /* Was it a request for an allowed domain? */
  if (this->iphash && 
      (!iphash_getip(this->iphash, NULL, (struct in_addr*) &pack->iph.daddr)))
    return 0;
  
  /* Was it a request for an allowed network? */
  for (i=0; i<this->uamoknetlen; i++) {
    if (this->uamokaddr[i].s_addr == 
	(pack->iph.daddr & this->uamokmask[i].s_addr))
      return 0;
  }

  /* Was it a http request for another server? */
  /* We are changing dest IP and dest port to local UAM server */
  if ((pack->iph.protocol == DHCP_IP_TCP) &&
      (tcph->dst == htons(DHCP_HTTP))) {
    conn->dnatip   = pack->iph.daddr; /* Save for undoing */
    conn->dnatport = tcph->src;       /* Save for undoing */
    pack->iph.daddr = this->uamlisten.s_addr;
    tcph->dst  = htons(this->uamport);
    dhcp_tcp_check(pack, len);
    dhcp_ip_check((struct dhcp_ippacket_t*) pack);
    return 0;
  }

  return -1; /* Something else */

}

/**
 * dhcp_undoDNAT()
 * Change destination address to authentication server.
 **/
int dhcp_undoDNAT(struct dhcp_conn_t *conn, 
		  struct dhcp_ippacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_tcphdr_t *tcph = (struct dhcp_tcphdr_t*) pack->payload;
  struct dhcp_udphdr_t *udph = (struct dhcp_udphdr_t*) pack->payload;
  int i;

  
  /* Was it a DNS reply? */
  if (((this->anydns) ||
       (pack->iph.saddr == conn->dns1.s_addr) ||
       (pack->iph.saddr == conn->dns2.s_addr)) &&
      (pack->iph.protocol == DHCP_IP_UDP) &&
      (udph->src == htons(DHCP_DNS)))
    return 0; 

  /* TODO Was it a DNAT http reply from redir server? */
  if ((pack->iph.saddr == this->uamlisten.s_addr) &&
      (pack->iph.protocol == DHCP_IP_TCP) &&
      (tcph->src == htons(this->uamport)) &&
      (tcph->dst == conn->dnatport)) {
    pack->iph.saddr = conn->dnatip;
    tcph->src = htons(DHCP_HTTP);
    dhcp_tcp_check(pack, len);
    dhcp_ip_check((struct dhcp_ippacket_t*) pack);
    return 0;
  }

  /* Was it a normal reply from redir server? */
  if ((pack->iph.saddr == this->uamlisten.s_addr) &&
      (pack->iph.protocol == DHCP_IP_TCP) &&
      (tcph->src == htons(this->uamport)))
    return 0; /* Destination was authentication server */

  /* Was it a normal http or https reply from authentication server? */
  for (i = 0; i<this->authiplen; i++) {
    if ((pack->iph.saddr == this->authip[i].s_addr) &&
	(pack->iph.protocol == DHCP_IP_TCP) &&
	((tcph->src == htons(DHCP_HTTP)) ||
	 (tcph->src == htons(DHCP_HTTPS))))
      return 0; /* Destination was authentication server */
  }

  /* Was it a reply from an allowed domain? */
  if (this->iphash && 
      (!iphash_getip(this->iphash, NULL, (struct in_addr*) &pack->iph.saddr)))
    return 0;

  /* Was it a reply from for an allowed network? */
  for (i=0; i<this->uamoknetlen; i++) {
    if (this->uamokaddr[i].s_addr == 
	(pack->iph.saddr & this->uamokmask[i].s_addr))
      return 0;
  }


  return -1; /* Something else */

}



/**
 * dhcp_getdefault()
 * Fill in a DHCP packet with most essential values
 **/
int
dhcp_getdefault(struct dhcp_fullpacket_t *pack)
{

  /* Initialise reply packet with request */
  memset(pack, 0, sizeof(struct dhcp_fullpacket_t));

  /* DHCP Payload */
  pack->dhcp.op     = DHCP_BOOTREPLY;
  pack->dhcp.htype  = DHCP_HTYPE_ETH;
  pack->dhcp.hlen   = DHCP_ETH_ALEN;

  /* UDP header */
  pack->udph.src = htons(DHCP_BOOTPS);
  pack->udph.dst = htons(DHCP_BOOTPC);

  /* IP header */
  pack->iph.ihl = 5;
  pack->iph.version = 4;
  pack->iph.tos = 0;
  pack->iph.tot_len = 0; /* Calculate at end of packet */
  pack->iph.id = 0;
  pack->iph.frag_off = 0;
  pack->iph.ttl = 0x10;
  pack->iph.protocol = 0x11;
  pack->iph.check = 0; /* Calculate at end of packet */

  /* Ethernet header */
  pack->ethh.prot = htons(DHCP_ETH_IP);

  return 0;
}


/**
 * dhcp_gettag()
 * Search a DHCP packet for a particular tag.
 * Returns -1 if not found.
 **/
int dhcp_gettag(struct dhcp_packet_t *pack, int length,
		struct dhcp_tag_t **tag, uint8_t tagtype) {
  struct dhcp_tag_t *t;
  int offset = DHCP_MIN_LEN + DHCP_OPTION_MAGIC_LEN;

  if ((length) >  DHCP_LEN)
    return -1;
  
  while ((offset + 2) < length) {
    t = (struct dhcp_tag_t*) (((void*) pack) + offset);
    if (t->t == tagtype) {
      if ((offset +  2 + t->l) > length)
	return -1; /* Tag length too long */
      *tag = t;
      return 0;
    }
    offset +=  2 + t->l;
  }
  
  return -1; /* Not found  */
  
}


/**
 * dhcp_sendOFFER()
 * Send of a DHCP offer message to a peer.
 **/
int dhcp_sendOFFER(struct dhcp_conn_t *conn, 
		   struct dhcp_fullpacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_fullpacket_t packet;
  uint16_t length = 576 + 4; /* Maximum length */
  uint16_t udp_len = 576 - 20; /* Maximum length */
  int pos = 0;

  struct sockaddr_ll dest;

  /* Get packet default values */
  dhcp_getdefault(&packet);

  
  /* DHCP Payload */
  packet.dhcp.xid    = pack->dhcp.xid;
  packet.dhcp.yiaddr = conn->hisip.s_addr;
  packet.dhcp.flags  = pack->dhcp.flags;
  packet.dhcp.giaddr = pack->dhcp.giaddr;
  memcpy(&packet.dhcp.chaddr, &pack->dhcp.chaddr, DHCP_CHADDR_LEN);

  /* Magic cookie */
  packet.dhcp.options[pos++] = 0x63;
  packet.dhcp.options[pos++] = 0x82;
  packet.dhcp.options[pos++] = 0x53;
  packet.dhcp.options[pos++] = 0x63;

  packet.dhcp.options[pos++] = DHCP_OPTION_MESSAGE_TYPE;
  packet.dhcp.options[pos++] = 1;
  packet.dhcp.options[pos++] = DHCPOFFER;

  packet.dhcp.options[pos++] = DHCP_OPTION_SUBNET_MASK;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->hismask.s_addr, 4);
  pos += 4;

  packet.dhcp.options[pos++] = DHCP_OPTION_ROUTER_OPTION;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->ourip.s_addr, 4);
  pos += 4;

  /* Insert DNS Servers if given */
  if (conn->dns1.s_addr && conn->dns2.s_addr) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DNS;
    packet.dhcp.options[pos++] = 8;
    memcpy(&packet.dhcp.options[pos], &conn->dns1.s_addr, 4);
    pos += 4;
    memcpy(&packet.dhcp.options[pos], &conn->dns2.s_addr, 4);
    pos += 4;
  }
  else if (conn->dns1.s_addr) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DNS;
    packet.dhcp.options[pos++] = 4;
    memcpy(&packet.dhcp.options[pos], &conn->dns1.s_addr, 4);
    pos += 4;
  }
  else if (conn->dns2.s_addr) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DNS;
    packet.dhcp.options[pos++] = 4;
    memcpy(&packet.dhcp.options[pos], &conn->dns2.s_addr, 4);
    pos += 4;
  }

  /* Insert Domain Name if present */
  if (strlen(conn->domain)) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DOMAIN_NAME;
    packet.dhcp.options[pos++] = strlen(conn->domain);
    memcpy(&packet.dhcp.options[pos], &conn->domain, strlen(conn->domain));
    pos += strlen(conn->domain);
  }

  packet.dhcp.options[pos++] = DHCP_OPTION_LEASE_TIME;
  packet.dhcp.options[pos++] = 4;
  packet.dhcp.options[pos++] = (this->lease >> 24) & 0xFF;
  packet.dhcp.options[pos++] = (this->lease >> 16) & 0xFF;
  packet.dhcp.options[pos++] = (this->lease >>  8) & 0xFF;
  packet.dhcp.options[pos++] = (this->lease >>  0) & 0xFF;

  /* Must be listening address */
  packet.dhcp.options[pos++] = DHCP_OPTION_SERVER_ID;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->ourip.s_addr, 4);
  pos += 4;

  packet.dhcp.options[pos++] = DHCP_OPTION_END;

  /* UDP header */
  udp_len = pos + DHCP_MIN_LEN + DHCP_UDP_HLEN;
  packet.udph.len = htons(udp_len);

  /* IP header */
  packet.iph.tot_len = htons(udp_len + DHCP_IP_HLEN);
  packet.iph.daddr = ~0; /* TODO: Always sending to broadcast address */
  packet.iph.saddr = conn->ourip.s_addr;

  /* Work out checksums */
  dhcp_udp_check(&packet);
  dhcp_ip_check((struct dhcp_ippacket_t*) &packet); 

  /* Ethernet header */
  memcpy(packet.ethh.dst, conn->hismac, ETH_ALEN);
  memcpy(packet.ethh.src, this->hwaddr, ETH_ALEN);

  /* Calculate total length */
  length = udp_len + DHCP_IP_HLEN + DHCP_ETH_HLEN;

  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));

  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_IP);
  dest.sll_ifindex = this->ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->fd, &packet, (length), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->fd, length);
    return -1;
  }
  return 0;
}

/**
 * dhcp_sendACK()
 * Send of a DHCP acknowledge message to a peer.
 **/
int dhcp_sendACK(struct dhcp_conn_t *conn, 
		 struct dhcp_fullpacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_fullpacket_t packet;
  uint16_t length = 576 + 4; /* Maximum length */
  uint16_t udp_len = 576 - 20; /* Maximum length */
  int pos = 0;

  struct sockaddr_ll dest;

  /* Get packet default values */
  dhcp_getdefault(&packet);

  
  /* DHCP Payload */
  packet.dhcp.xid    = pack->dhcp.xid;
  packet.dhcp.ciaddr = pack->dhcp.ciaddr;
  packet.dhcp.yiaddr = conn->hisip.s_addr;
  packet.dhcp.flags  = pack->dhcp.flags;
  packet.dhcp.giaddr = pack->dhcp.giaddr;
  memcpy(&packet.dhcp.chaddr, &pack->dhcp.chaddr, DHCP_CHADDR_LEN);

  /* Magic cookie */
  packet.dhcp.options[pos++] = 0x63;
  packet.dhcp.options[pos++] = 0x82;
  packet.dhcp.options[pos++] = 0x53;
  packet.dhcp.options[pos++] = 0x63;

  packet.dhcp.options[pos++] = DHCP_OPTION_MESSAGE_TYPE;
  packet.dhcp.options[pos++] = 1;
  packet.dhcp.options[pos++] = DHCPACK;

  packet.dhcp.options[pos++] = DHCP_OPTION_SUBNET_MASK;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->hismask.s_addr, 4);
  pos += 4;

  packet.dhcp.options[pos++] = DHCP_OPTION_ROUTER_OPTION;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->ourip.s_addr, 4);
  pos += 4;

  /* Insert DNS Servers if given */
  if (conn->dns1.s_addr && conn->dns2.s_addr) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DNS;
    packet.dhcp.options[pos++] = 8;
    memcpy(&packet.dhcp.options[pos], &conn->dns1.s_addr, 4);
    pos += 4;
    memcpy(&packet.dhcp.options[pos], &conn->dns2.s_addr, 4);
    pos += 4;
  }
  else if (conn->dns1.s_addr) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DNS;
    packet.dhcp.options[pos++] = 4;
    memcpy(&packet.dhcp.options[pos], &conn->dns1.s_addr, 4);
    pos += 4;
  }
  else if (conn->dns2.s_addr) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DNS;
    packet.dhcp.options[pos++] = 4;
    memcpy(&packet.dhcp.options[pos], &conn->dns2.s_addr, 4);
    pos += 4;
  }

  /* Insert Domain Name if present */
  if (strlen(conn->domain)) {
    packet.dhcp.options[pos++] = DHCP_OPTION_DOMAIN_NAME;
    packet.dhcp.options[pos++] = strlen(conn->domain);
    memcpy(&packet.dhcp.options[pos], &conn->domain, strlen(conn->domain));
    pos += strlen(conn->domain);
  }

  packet.dhcp.options[pos++] = DHCP_OPTION_LEASE_TIME;
  packet.dhcp.options[pos++] = 4;
  packet.dhcp.options[pos++] = (this->lease >> 24) & 0xFF;
  packet.dhcp.options[pos++] = (this->lease >> 16) & 0xFF;
  packet.dhcp.options[pos++] = (this->lease >>  8) & 0xFF;
  packet.dhcp.options[pos++] = (this->lease >>  0) & 0xFF;

  /*
  packet.dhcp.options[pos++] = DHCP_OPTION_INTERFACE_MTU;
  packet.dhcp.options[pos++] = 2;
  packet.dhcp.options[pos++] = (conn->mtu >> 8) & 0xFF;
  packet.dhcp.options[pos++] = (conn->mtu >> 0) & 0xFF;
  */

  /* Must be listening address */
  packet.dhcp.options[pos++] = DHCP_OPTION_SERVER_ID;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->ourip.s_addr, 4);
  pos += 4;

  packet.dhcp.options[pos++] = DHCP_OPTION_END;

  /* UDP header */
  udp_len = pos + DHCP_MIN_LEN + DHCP_UDP_HLEN;
  packet.udph.len = htons(udp_len);

  /* IP header */
  packet.iph.tot_len = htons(udp_len + DHCP_IP_HLEN);
  packet.iph.daddr = ~0; /* TODO: Always sending to broadcast address */
  packet.iph.saddr = conn->ourip.s_addr;

  /* Work out checksums */
  dhcp_udp_check(&packet);
  dhcp_ip_check((struct dhcp_ippacket_t*) &packet); 

  /* Ethernet header */
  memcpy(packet.ethh.dst, conn->hismac, ETH_ALEN);
  memcpy(packet.ethh.src, this->hwaddr, ETH_ALEN);

  /* Calculate total length */
  length = udp_len + DHCP_IP_HLEN + DHCP_ETH_HLEN;


  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));

  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_IP);
  dest.sll_ifindex = this->ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->fd, &packet, (length), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->fd, length);
    return -1;
  }
  return 0;
}

/**
 * dhcp_sendNAK()
 * Send of a DHCP negative acknowledge message to a peer.
 * NAK messages are always sent to broadcast IP address (
 * except when using a DHCP relay server)
 **/
int dhcp_sendNAK(struct dhcp_conn_t *conn, 
		 struct dhcp_fullpacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_fullpacket_t packet;
  uint16_t length = 576 + 4; /* Maximum length */
  uint16_t udp_len = 576 - 20; /* Maximum length */
  int pos = 0;

  struct sockaddr_ll dest;

  /* Get packet default values */
  dhcp_getdefault(&packet);

  
  /* DHCP Payload */
  packet.dhcp.xid    = pack->dhcp.xid;
  packet.dhcp.flags  = pack->dhcp.flags;
  packet.dhcp.giaddr = pack->dhcp.giaddr;
  memcpy(&packet.dhcp.chaddr, &pack->dhcp.chaddr, DHCP_CHADDR_LEN);

  /* Magic cookie */
  packet.dhcp.options[pos++] = 0x63;
  packet.dhcp.options[pos++] = 0x82;
  packet.dhcp.options[pos++] = 0x53;
  packet.dhcp.options[pos++] = 0x63;

  packet.dhcp.options[pos++] = DHCP_OPTION_MESSAGE_TYPE;
  packet.dhcp.options[pos++] = 1;
  packet.dhcp.options[pos++] = DHCPNAK;

  /* Must be listening address */
  packet.dhcp.options[pos++] = DHCP_OPTION_SERVER_ID;
  packet.dhcp.options[pos++] = 4;
  memcpy(&packet.dhcp.options[pos], &conn->ourip.s_addr, 4);
  pos += 4;

  packet.dhcp.options[pos++] = DHCP_OPTION_END;

  /* UDP header */
  udp_len = pos + DHCP_MIN_LEN + DHCP_UDP_HLEN;
  packet.udph.len = htons(udp_len);

  /* IP header */
  packet.iph.tot_len = htons(udp_len + DHCP_IP_HLEN);
  packet.iph.daddr = ~0; /* TODO: Always sending to broadcast address */
  packet.iph.saddr = conn->ourip.s_addr;

  /* Work out checksums */
  dhcp_udp_check(&packet);
  dhcp_ip_check((struct dhcp_ippacket_t*) &packet); 

  /* Ethernet header */
  memcpy(packet.ethh.dst, conn->hismac, ETH_ALEN);
  memcpy(packet.ethh.src, this->hwaddr, ETH_ALEN);

  /* Calculate total length */
  length = udp_len + DHCP_IP_HLEN + DHCP_ETH_HLEN;

  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));

  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_IP);
  dest.sll_ifindex = this->ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->fd, &packet, (length), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->fd, length);
    return -1;
  }
  return 0;
}


/**
 * dhcp_getreq()
 * Process a received DHCP request MESSAGE.
 **/
int dhcp_getreq(struct dhcp_t *this, 
		struct dhcp_fullpacket_t *pack, int len) {
  struct dhcp_conn_t *conn;

  struct dhcp_tag_t *message_type = 0;
  struct dhcp_tag_t *requested_ip = 0;

  if (pack->udph.dst != htons(DHCP_BOOTPS)) 
    return 0; /* Not a DHCP packet */

  if (dhcp_gettag(&pack->dhcp, ntohs(pack->udph.len)-DHCP_UDP_HLEN, 
		  &message_type, DHCP_OPTION_MESSAGE_TYPE)) {
    return -1;
  }

  if (message_type->l != 1)
    return -1; /* Wrong length of message type */

  if ((message_type->v[0] != DHCPDISCOVER) && 
      (message_type->v[0] != DHCPREQUEST) &&
      (message_type->v[0] != DHCPRELEASE)) {
    return 0; /* Unsupported message type */
  }

  /* Release message */
  /* If connection exists: Release it. No Reply to client is sent */
  if (message_type->v[0] == DHCPRELEASE) {
    if (!dhcp_hashget(this, &conn, pack->ethh.src)) {
      dhcp_freeconn(conn);
    }
    return 0;
  }

  /* Check to see if we know MAC address. If not allocate new conn */
  if (dhcp_hashget(this, &conn, pack->ethh.src)) {

    /* Do we allow dynamic allocation of IP addresses? */
    if (!this->allowdyn) 
      return 0; 

    /* Allocate new connection */
    if (dhcp_newconn(this, &conn, pack->ethh.src))
      return 0; /* Out of connections */
  }

  /* Request an IP address */
  if (conn->authstate == DHCP_AUTH_NONE) {
    if (this ->cb_request)
      if (this->cb_request(conn)) {
	return 0; /* Ignore request if IP address was not allocated */
      }
  }

  gettimeofday(&conn->lasttime, NULL);

  /* Discover message */
  /* If an IP address was assigned offer it to the client */
  /* Otherwise ignore the request */
  if (message_type->v[0] == DHCPDISCOVER) {
    if (conn->hisip.s_addr) dhcp_sendOFFER(conn, pack, len);
    return 0;
  }
  
  /* Request message */
  if (message_type->v[0] == DHCPREQUEST) {

    if (!conn->hisip.s_addr) {
      if (this->debug) printf("hisip not set\n");
      return dhcp_sendNAK(conn, pack, len);
    }

    if (!memcmp(&conn->hisip.s_addr, &pack->dhcp.ciaddr, 4)) {
      if (this->debug) printf("hisip match ciaddr\n");
      return dhcp_sendACK(conn, pack, len);
    }

    if (!dhcp_gettag(&pack->dhcp, ntohs(pack->udph.len)-DHCP_UDP_HLEN, 
		    &requested_ip, DHCP_OPTION_REQUESTED_IP)) {
      if (!memcmp(&conn->hisip.s_addr, requested_ip->v, 4))
	return dhcp_sendACK(conn, pack, len);
    }

    return dhcp_sendNAK(conn, pack, len);
  }
  
  /* Unsupported DHCP message: Ignore */
  return 0;
}


/**
 * dhcp_set_addrs()
 * Set various IP addresses of a connection.
 **/
int dhcp_set_addrs(struct dhcp_conn_t *conn,
		   struct in_addr *hisip,
		   struct in_addr *hismask,
		   struct in_addr *ourip,
		   struct in_addr *dns1,
		   struct in_addr *dns2,
		   char *domain) {

  conn->hisip.s_addr = hisip->s_addr;
  conn->hismask.s_addr = hismask->s_addr;
  conn->ourip.s_addr = ourip->s_addr;
  conn->dns1.s_addr = dns1->s_addr;
  conn->dns2.s_addr = dns2->s_addr;

  if (domain) {
    strncpy(conn->domain, domain, DHCP_DOMAIN_LEN);
    conn->domain[DHCP_DOMAIN_LEN-1] = 0;
  }
  else {
    conn->domain[0] = 0;
  }
  
  return 0;
}


/**
 * dhcp_decaps()
 * Call this function when a new IP packet has arrived. This function
 * should be part of a select() loop in the application.
 **/
int dhcp_decaps(struct dhcp_t *this)  /* DHCP Indication */
{
  struct dhcp_ippacket_t packet;
  int length;
  
  struct dhcp_conn_t *conn;
  struct in_addr ourip;
  unsigned char const bmac[ETH_ALEN] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

  if (this->debug) printf("DHCP packet received\n");

  if ((length = recv(this->fd, &packet, sizeof(packet), 0)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "recv(fd=%d, len=%d) failed",
	    this->fd, sizeof(packet));
    return -1;
  }

  /* Check that MAC address is our MAC or Broadcast */
  if ((memcmp(packet.ethh.dst, this->hwaddr, ETH_ALEN)) && (memcmp(packet.ethh.dst, bmac, ETH_ALEN)))
    return 0;

  /* Check to see if we know MAC address. */
  if (!dhcp_hashget(this, &conn, packet.ethh.src)) {
    if (this->debug) printf("Address found\n");
    ourip.s_addr = conn->ourip.s_addr;
  }
  else {
    if (this->debug) printf("Address not found\n");
    ourip.s_addr = 0;
  }

  /* Check to see if it is a packet for us */
  /* TODO: Handle IP packets with options. Currently these are just ignored */
  if ((packet.iph.daddr == 0) ||
      (packet.iph.daddr == 0xffffffff) ||
      (packet.iph.daddr == ourip.s_addr)) {

    /* if (packet.iph.ihl != 5)
      return 0;  Packets for us with IP options are silently dropped */

    /* if (packet.iph.protocol != DHCP_IP_UDP)
      return 0;   Packets for us which are not UDP are silently dropped */
    
    /* We handle DHCP IPv4 packets without header options */
    if ((packet.iph.ihl == 5) && (packet.iph.protocol == DHCP_IP_UDP)) {
      dhcp_getreq(this, (struct dhcp_fullpacket_t*) &packet, length);
      return 0; /* TODO */
    }
  }


  /* Return if we do not know peer */
  if (!conn) 
    return 0;

  gettimeofday(&conn->lasttime, NULL);

  switch (conn->authstate) {
  case DHCP_AUTH_UAM:
  case DHCP_AUTH_WPA:
    /* Pass packets unmodified */
    break; 
  case DHCP_AUTH_UNAUTH_TOS:
    /* Set TOS to specified value (unauthenticated) */
    packet.iph.tos = conn->unauth_cp;
    dhcp_ip_check(&packet);
    break;
  case DHCP_AUTH_AUTH_TOS:
    /* Set TOS to specified value (authenticated) */
    packet.iph.tos = conn->auth_cp;
    dhcp_ip_check(&packet);
    break;
  case DHCP_AUTH_DNAT:
    /* Destination NAT if request to unknown web server */
    if (dhcp_doDNAT(conn, &packet, length))
      return 0; /* Drop is not http or dns */
    break;
  case DHCP_AUTH_DROP: 
  default:
    return 0;
  }

  if ((conn->hisip.s_addr) && (this ->cb_data_ind)) {
    this ->cb_data_ind(conn, &packet.iph, length-DHCP_ETH_HLEN);
  }
  
  return 0;
}

/**
 * dhcp_data_req()
 * Call this function to send an IP packet to the peer.
 **/
int dhcp_data_req(struct dhcp_conn_t *conn, void *pack, unsigned len)
{
  struct dhcp_t *this = conn->parent;
  int length = len + DHCP_ETH_HLEN;
  struct sockaddr_ll dest;
  
  struct dhcp_ippacket_t packet;


  /* Ethernet header */
  memcpy(packet.ethh.dst, conn->hismac, ETH_ALEN);
  memcpy(packet.ethh.src, this->hwaddr, ETH_ALEN);
  packet.ethh.prot = htons(DHCP_ETH_IP);

  /* IP Packet */
  memcpy(&packet.iph, pack, len);
  
  switch (conn->authstate) {
  case DHCP_AUTH_UAM:
  case DHCP_AUTH_WPA:
  case DHCP_AUTH_UNAUTH_TOS:
  case DHCP_AUTH_AUTH_TOS:
    /* Pass packets unmodified */
    break; 
  case DHCP_AUTH_DNAT:
    /* Undo destination NAT */
    if (dhcp_undoDNAT(conn, &packet, length))
      return 0;
    break;
  case DHCP_AUTH_DROP: 
  default:
    return 0;
  }

  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));

  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_IP);
  dest.sll_ifindex = this->ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->fd, &packet, (length), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->fd, length);
    return -1;
  }
  return 0;

}


/**
 * dhcp_sendARP()
 * Send ARP message to peer
 **/
int dhcp_sendARP(struct dhcp_conn_t *conn, 
		 struct dhcp_arp_fullpacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_arp_fullpacket_t packet;
  uint16_t length = sizeof(packet);
  struct in_addr reqaddr;
  struct sockaddr_ll dest;

  /* Get local copy */
  memcpy(&reqaddr.s_addr, pack->arp.tpa, DHCP_IP_ALEN);

  /* Check that request is within limits */

  
  /* Is ARP request for clients own address: Ignore */
  if (conn->hisip.s_addr == reqaddr.s_addr)
    return 0;

  /* If ARP request outside of mask: Ignore */
  if ((conn->hisip.s_addr & conn->hismask.s_addr) !=
      (reqaddr.s_addr & conn->hismask.s_addr))
    return 0;

  /* Get packet default values */
  memset(&packet, 0, sizeof(packet));
	 
  /* ARP Payload */
  packet.arp.hrd = htons(DHCP_HTYPE_ETH);
  packet.arp.pro = htons(DHCP_ETH_IP);
  packet.arp.hln = DHCP_ETH_ALEN;
  packet.arp.pln = DHCP_IP_ALEN;
  packet.arp.op  = htons(DHCP_ARP_REPLY);

  /* Source address */
  memcpy(packet.arp.sha, this->arp_hwaddr, DHCP_ETH_ALEN);
  memcpy(packet.arp.spa, &reqaddr.s_addr, DHCP_IP_ALEN);

  /* Target address */
  memcpy(packet.arp.tha, &conn->hismac, DHCP_ETH_ALEN);
  memcpy(packet.arp.tpa, &conn->hisip.s_addr, DHCP_IP_ALEN);
  

  /* Ethernet header */
  memcpy(packet.ethh.dst, conn->hismac, ETH_ALEN);
  memcpy(packet.ethh.src, this->hwaddr, ETH_ALEN);
  packet.ethh.prot = htons(DHCP_ETH_ARP);

  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));

  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_ARP);
  dest.sll_ifindex = this->arp_ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->arp_fd, &packet, (length), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->arp_fd, length);
    return -1;
  }
  return 0;
}


/**
 * dhcp_arp_ind()
 * Call this function when a new ARP packet has arrived. This function
 * should be part of a select() loop in the application.
 **/
int dhcp_arp_ind(struct dhcp_t *this)  /* ARP Indication */
{
  struct dhcp_arp_fullpacket_t packet;
  int length;
  
  struct dhcp_conn_t *conn;
  unsigned char const bmac[ETH_ALEN] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

  if (this->debug) printf("ARP Packet Received\n");

  if ((length = recv(this->arp_fd, &packet, sizeof(packet), 0)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "recv(fd=%d, len=%d) failed",
	    this->arp_fd, sizeof(packet));
    return -1;
  }


  /* Check that MAC address is our MAC or Broadcast */
  if ((memcmp(packet.ethh.dst, this->hwaddr, ETH_ALEN)) && (memcmp(packet.ethh.dst, bmac, ETH_ALEN)))
    return 0;

  /* Check to see if we know MAC address. */
  if (!dhcp_hashget(this, &conn, packet.ethh.src)) {
    if (this->debug) printf("Address found\n");
  }
  else {
    if (this->debug) printf("Address not found\n");
    return 0;
  }
  
  gettimeofday(&conn->lasttime, NULL);

  if (!conn->hisip.s_addr)
    return 0; /* Only reply if he was allocated an address */

  if (!memcmp(&conn->hisip.s_addr, packet.arp.tpa, 4))
    return 0; /* Only reply if he asked for his router address */
  
  dhcp_sendARP(conn, &packet, length);

  return 0;
}




/**
 * eapol_sendNAK()
 * Send of a EAPOL negative acknowledge message to a peer.
 * NAK messages are always sent to broadcast IP address (
 * except when using a EAPOL relay server)
 **/
int dhcp_senddot1x(struct dhcp_conn_t *conn,  
		   struct dhcp_dot1xpacket_t *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct sockaddr_ll dest;
  
  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));
  
  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_EAPOL);
  dest.sll_ifindex = this->ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->fd, pack, (len), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->fd, len);
    return -1;
  }
  return 0;
}

/**
 * eapol_sendNAK()
 * Send of a EAPOL negative acknowledge message to a peer.
 * NAK messages are always sent to broadcast IP address (
 * except when using a EAPOL relay server)
 **/
int dhcp_sendEAP(struct dhcp_conn_t *conn, void *pack, int len) {

  struct dhcp_t *this = conn->parent;
  struct dhcp_dot1xpacket_t packet;

  struct sockaddr_ll dest;

  /* Ethernet header */
  memcpy(packet.ethh.dst, conn->hismac, ETH_ALEN);
  memcpy(packet.ethh.src, this->hwaddr, ETH_ALEN);
  packet.ethh.prot = htons(DHCP_ETH_EAPOL);
  
  /* 802.1x header */
  packet.dot1x.ver  = 1;
  packet.dot1x.type = 0; /* EAP */
  packet.dot1x.len =  ntohs(len);

  memcpy(&packet.eap, pack, len);
  
  /* setting up sockaddr_ll */
  memset(&dest, '\0', sizeof(dest));

  /* destinatin address for sendto */
  dest.sll_family = AF_PACKET;
  dest.sll_protocol = htons(DHCP_ETH_EAPOL);
  dest.sll_ifindex = this->ifindex;
  dest.sll_halen = ETH_ALEN;
  memcpy (dest.sll_addr, conn->hismac, ETH_ALEN);
  if (sendto(this->fd, &packet, (DHCP_ETH_HLEN + 4 + len), 0,
	     (struct sockaddr *)&dest ,sizeof(dest)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "sendto(fd=%d, len=%d) failed",
	    this->fd, len);
    return -1;
  }
  return 0;
}

int dhcp_sendEAPreject(struct dhcp_conn_t *conn, void *pack, int len) {

  /*struct dhcp_t *this = conn->parent;*/

  struct dhcp_eap_t packet;


  if (pack) {
    dhcp_sendEAP(conn, pack, len);
  }
  else {
    memset(&packet, 0, sizeof(pack));
    packet.code      =  4;
    packet.id        =  1; /* TODO ??? */
    packet.length    =  ntohs(4);
  
    dhcp_sendEAP(conn, &packet, 4);
  }

  return 0;

}


/**
 * dhcp_eapol_ind()
 * Call this function when a new EAPOL packet has arrived. This function
 * should be part of a select() loop in the application.
 **/
int dhcp_eapol_ind(struct dhcp_t *this)  /* EAPOL Indication */
{
  struct dhcp_dot1xpacket_t packet;
  int length;
  
  struct dhcp_conn_t *conn = NULL;
  unsigned char const bmac[ETH_ALEN] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
  unsigned char const amac[ETH_ALEN] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};

  if (this->debug) printf("EAPOL packet received\n");
  
  if ((length = recv(this->eapol_fd, &packet, sizeof(packet), 0)) < 0) {
    sys_err(LOG_ERR, __FILE__, __LINE__, errno,
	    "recv(fd=%d, len=%d) failed",
	    this->fd, sizeof(packet));
    return -1;
  }
  
  /* Check to see if we know MAC address. */
  if (!dhcp_hashget(this, &conn, packet.ethh.src)) {
    if (this->debug) printf("Address found\n");
  }
  else {
    if (this->debug) printf("Address not found\n");
  }
  
  /* Check that MAC address is our MAC, Broadcast or authentication MAC */
  if ((memcmp(packet.ethh.dst, this->hwaddr, ETH_ALEN)) && (memcmp(packet.ethh.dst, bmac, ETH_ALEN)) && (memcmp(packet.ethh.dst, amac, ETH_ALEN)))
    return 0;
  
  if (this->debug) printf("IEEE 802.1x Packet: %.2x, %.2x %d\n",
			  packet.dot1x.ver, packet.dot1x.type,
			  ntohs(packet.dot1x.len));
  
  if (packet.dot1x.type == 1) { /* Start */
    struct dhcp_dot1xpacket_t pack;
    memset(&pack, 0, sizeof(pack));
    
    /* Allocate new connection */
    if (conn == NULL) {
      if (dhcp_newconn(this, &conn, packet.ethh.src))
	return 0; /* Out of connections */
    }

    /* Ethernet header */
    memcpy(pack.ethh.dst, packet.ethh.src, ETH_ALEN);
    memcpy(pack.ethh.src, this->hwaddr, ETH_ALEN);
    pack.ethh.prot = htons(DHCP_ETH_EAPOL);

    /* 802.1x header */
    pack.dot1x.ver  = 1;
    pack.dot1x.type = 0; /* EAP */
    pack.dot1x.len =  ntohs(5);
    
    /* EAP Packet */
    pack.eap.code      =  1;
    pack.eap.id        =  1;
    pack.eap.length    =  ntohs(5);
    pack.eap.type      =  1; /* Identity */

    dhcp_senddot1x(conn, &pack, DHCP_ETH_HLEN + 4 + 5);
    return 0;
  }
  else if (packet.dot1x.type == 0) { /* EAP */

    /* TODO: Currently we only support authentications starting with a
       client sending a EAPOL start message. Need to also support
       authenticator initiated communications. */
    if (!conn)
      return 0;

    gettimeofday(&conn->lasttime, NULL);
    
    if (this ->cb_eap_ind)
      this ->cb_eap_ind(conn, &packet.eap, ntohs(packet.eap.length));
    return 0;
  }
  else { /* Check for logoff */
    return 0;
  }
}


/**
 * dhcp_set_cb_eap_ind()
 * Set callback function which is called when packet has arrived
 * Used for eap packets
 **/
extern int dhcp_set_cb_eap_ind(struct dhcp_t *this, 
  int (*cb_eap_ind) (struct dhcp_conn_t *conn, void *pack, unsigned len))
{
  this ->cb_eap_ind = cb_eap_ind;
  return 0;
}


/**
 * dhcp_set_cb_data_ind()
 * Set callback function which is called when packet has arrived
 **/
extern int dhcp_set_cb_data_ind(struct dhcp_t *this, 
  int (*cb_data_ind) (struct dhcp_conn_t *conn, void *pack, unsigned len))
{
  this ->cb_data_ind = cb_data_ind;
  return 0;
}


/**
 * dhcp_set_cb_data_ind()
 * Set callback function which is called when a dhcp request is received
 **/
extern int dhcp_set_cb_request(struct dhcp_t *this, 
  int (*cb_request) (struct dhcp_conn_t *conn))
{
  this ->cb_request = cb_request;
  return 0;
}


/**
 * dhcp_set_cb_connect()
 * Set callback function which is called when a connection is created
 **/
extern int dhcp_set_cb_connect(struct dhcp_t *this, 
  int (*cb_connect) (struct dhcp_conn_t *conn))
{
  this ->cb_connect = cb_connect;
  return 0;
}

/**
 * dhcp_set_cb_disconnect()
 * Set callback function which is called when a connection is deleted
 **/
extern int dhcp_set_cb_disconnect(struct dhcp_t *this, 
  int (*cb_disconnect) (struct dhcp_conn_t *conn))
{
  this ->cb_disconnect = cb_disconnect;
  return 0;
}


