/* Kernel module to match connection tracking byte counter.
 * GPL (C) 2002 Martin Devera (devik@cdi.cz).
 */
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netfilter_ipv4/ip_conntrack.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include <linux/netfilter_ipv4/ipt_connbytes.h>

static int
match(const struct sk_buff *skb,
      const struct net_device *in,
      const struct net_device *out,
      const void *matchinfo,
      int offset,
      int *hotdrop)
{
	const struct ipt_connbytes_info *sinfo = matchinfo;
	enum ip_conntrack_info ctinfo;
	struct ip_conntrack *ct;

	if (!(ct = ip_conntrack_get((struct sk_buff *)skb, &ctinfo)))
		return 0; /* no match */
	
	if (sinfo->from > sinfo->to)
		return (ct->bytes < sinfo->to || ct->bytes > sinfo->from);
	else
		return (ct->bytes >= sinfo->from && ct->bytes <= sinfo->to);
}

static int check(const char *tablename,
		 const struct ipt_ip *ip,
		 void *matchinfo,
		 unsigned int matchsize,
		 unsigned int hook_mask)
{
	if (matchsize != IPT_ALIGN(sizeof(struct ipt_connbytes_info)))
		return 0;

	return 1;
}

static struct ipt_match state_match = {
	.name		= "connbytes",
	.match		= &match,
	.checkentry	= &check,
	.me		= THIS_MODULE
};

static int __init init(void)
{
	return ipt_register_match(&state_match);
}

static void __exit fini(void)
{
	ipt_unregister_match(&state_match);
}

module_init(init);
module_exit(fini);
MODULE_LICENSE("GPL");
